<?php

declare (strict_types=1);
namespace Syde\Vendor\Inpsyde\PayoneerForWoocommerce\EmbeddedPayment\PaymentProcessor;

use Exception;
use Syde\Vendor\Inpsyde\PayoneerForWoocommerce\Checkout\Authentication\TokenGeneratorInterface;
use Syde\Vendor\Inpsyde\PayoneerForWoocommerce\Checkout\MisconfigurationDetector\MisconfigurationDetectorInterface;
use Syde\Vendor\Inpsyde\PayoneerForWoocommerce\Checkout\PaymentProcessor\AbstractPaymentProcessor;
use Syde\Vendor\Inpsyde\PayoneerForWoocommerce\ListSession\ListSession\ListSessionProvider;
use Syde\Vendor\Inpsyde\PayoneerForWoocommerce\ListSession\ListSession\PaymentContext;
use Syde\Vendor\Inpsyde\PayoneerForWoocommerce\Api\Gateway\CommandFactory\WcOrderBasedUpdateCommandFactoryInterface;
use Syde\Vendor\Inpsyde\PaymentGateway\PaymentGateway;
use Syde\Vendor\Inpsyde\PayoneerSdk\Api\Command\Exception\CommandExceptionInterface;
use Syde\Vendor\Inpsyde\PayoneerSdk\Api\Command\Exception\InteractionExceptionInterface;
use WC_Order;
/**
 * @psalm-import-type PaymentResult from AbstractPaymentProcessor
 */
class EmbeddedPaymentProcessor extends AbstractPaymentProcessor
{
    /**
     * @var string
     */
    protected string $hostedModeOverrideFlag;
    protected bool $isRestRequest;
    public function __construct(WcOrderBasedUpdateCommandFactoryInterface $updateCommandFactory, ListSessionProvider $sessionProvider, TokenGeneratorInterface $tokenGenerator, string $tokenKey, string $transactionIdFieldName, string $hostedModeOverrideFlag, MisconfigurationDetectorInterface $misconfigurationDetector, string $checkoutSessionHashKey, bool $isRestRequest)
    {
        parent::__construct($misconfigurationDetector, $sessionProvider, $updateCommandFactory, $tokenGenerator, $tokenKey, $transactionIdFieldName, $checkoutSessionHashKey);
        $this->hostedModeOverrideFlag = $hostedModeOverrideFlag;
        $this->isRestRequest = $isRestRequest;
    }
    public function processPayment(WC_Order $order, PaymentGateway $gateway): array
    {
        $this->sessionProvider->provide(new PaymentContext($order));
        try {
            $result = parent::processPayment($order, $gateway);
        } catch (InteractionExceptionInterface $exception) {
            return $this->handleInteractionException($order, $exception);
        } catch (CommandExceptionInterface $exception) {
            $exceptionWrapper = new Exception(
                /* translators: An unexpected error during the final List UPDATE before the CHARGE */
                __('Payment failed. Please attempt the payment again or contact the shop admin. This issue has been logged.', 'payoneer-checkout'),
                $exception->getCode(),
                $exception
            );
            return $this->handleFailedPaymentProcessing($order, $exceptionWrapper);
        }
        /**
         * We always signal success: The actual payment is supposed to be handled by the JS WebSDK
         * or by the hosted payment page.
         * in the customer's browser session. Our 'redirect' URL is only a fallback in case our JS
         * is somehow broken. For this reason, we also add the flag to force hosted mode.
         * The WebSDK is taking care of redirecting to 'thank-you' after finishing the transaction.
         * If this somehow does not happen, we still instruct WC to move to the payment page
         *
         * But this doesn't work properly for block checkout.
         */
        if (!$this->isRestRequest) {
            $result['redirect'] = add_query_arg([$this->hostedModeOverrideFlag => \true], $order->get_checkout_payment_url());
        }
        /* translators: Order note added when processing an order in embedded flow */
        $note = __('Backend processing finished, frontend processing is about to start.', 'payoneer-checkout');
        $this->putOrderOnHold($order, $note . \PHP_EOL);
        return $result;
    }
}
